<?php
/***
*	Classe : TForm
*
*   Notes : La classe TForm est la classe qui g�re la g�n�ration des formulaires. C'est celle qu'il faut instancier dans tous les cas.
*
*   Licence : Ce code est enti�rement libre de droits. Vous pouvez le copier, le modifier, le partager, et ce sans aucune restriction
*
*   Auteur : Christophe TAFANI-DEREEPER <christophetd@gmail.com>
*   Version modifi�e par Alexandre Peyron
*
*   Date de cr�ation : 24 juillet 2009
*   Date de derni�re modification : 18 mai 2011
* 
*/

class TForm
{
	/***
	* Le nom unique du formulaire
	* @var protected string nom
	*/
	protected $nom;
	
	/***
	* La m�thode d'envoi des donn�es du formulaire (POST ou GET)
	* @var protected string methode
	*/
	protected $methode;
	
	
	/***
	 * Ajoute une classe au form (CSS)
	 * @var protected string methode
	 */
	protected $classCSS = "";
	
	/***
	* La page de traitement du formulaire. Par d�faut, aucune action n'est effectu�e si le fichier n'existe pas.
	* Modifier le param�tre $checkAction du constructeur pour.
	* @var protected string nom
	*/
	protected $action;
	
	/***
	* Indique si les donn�es sortantes du formulaire doivent �tre trait�es par htmlspecialchars()
	* @var protected bool protectHtml
	*
	*/
	protected $protectHtml;
	
	/***
	* L'array contenant les objets, �l�ments du formulaires, instanci�s depuis la m�thode add()
	* @var protected array elements
	*/
	private $elements = array();
	
	
	public $submit = 'Valider';

	/***
	* __construct
	*
	* Le constructeur de la classe ZForm
	*
	* @var string $nom = NULL				Le nom unique du formulaire
	* @var string $methode = POST			La m�thode utilis�e par le formulaire pour transf�rer les donn�es
	* @var string $action = NULL			La page de traitement du formulaire
	* @var bool $checkAction = FALSE		Le bol�en indiquant si le constructeur doit afficher une erreur en cas d'indication d'une page de traitement inexistante
	*/
	public function __construct($nom=NULL, $methode = 'POST', $action = NULL, $checkAction = false)
	{
		$this->nom = $nom;
		$this->methode = $methode;
		
		if($checkAction && !file_exists($action))
		{
			trigger_error('La page de traitement d�finie n\'existe pas.', E_USER_WARNING);
		}
		$this->action = $action;
		
		if(get_magic_quotes_gpc())
		{
			$_POST = array_map('stripslashes', $_POST);
			$_GET = array_map('stripslashes', $_GET);
			$_COOKIE = array_map('stripslashes', $_COOKIE);
		}
		
		$this->protectHtml = true;
	}

	/**
	* __toString()
	*
	* G�n�re le formulaire lors d'un echo de l'instance de la classe
	*
	* @return string       Contient le html du formulaire g�n�r�
	*/
	public function __toString()
	{
		if(empty($this->elements)) return 'Aucun élement dans le formulaire';
		
		$out = '';
		$out .= '<form action="'.$this->action.'" method="'.$this->methode.'" id="'.$this->nom.'" class="'.$this->classCSS.'" enctype="multipart/form-data">';
		$out .= '<ul>';
		foreach($this->elements as $element)
		{
			if(!$element->isHidden())
				$out .= '<li>'.$element->getHtml().'</li>';
			else	
				$out .= $element->getHtml();
		}
		$out .= '<li><input type="submit" name="submit" value="'.$this->submit.'" /><li>';
		$out .= '</ul>';
		$out .= '</form>';
		return $out;
	}
	
	/**
	* add($type)
	*
	*  Ajoute un champ en instanciant la classe correspondante et stocke l'objet instanci� dans le tableau elements
	*
	* @var string $type					   Indique la nature du champ � ajouter
	* @return mixed 					   Object si le type est reconnu par la classe, bool (FALSE) dans le cas contraire
	*/
	public function add($type)
	{
		$rempls = array('Text' => 'TFTextInput', 
						'Email' => 'TFMailInput', 
						'Date' => 'TFDateInput', 
						'Checkbox' => 'TFCheckBox',
						'Radio' => 'TFRadio',
						'Select' => 'TFSelectList',
						'Password' => 'TFPasswordInput',
						'Hidden' => 'TFHiddenField', 
						'Textarea' => 'TFTextareaInput', 
						'File' => 'TFileInput',
						'Separateur' => 'TFSeparateur');
		foreach($rempls as $i => $a)
		{
			$type = preg_replace('#^'.$i.'$#isU', $a, $type);
		}
		if(!($obj = new $type())) return false;
		$this->elements[] = $obj;
		return $obj;
	}
	
	/**
	* 
	*  Renvoie TRUE si le formulaire a �t� soumis, FALSE dans le cas contraire
	*
	* @return bool
	*/
	public function isSubmitted()
	{
		return (isset($_POST['submit']));
	}
	
	/**
	* 
	*  Renvoie TRUE si le formulaire a �t� soumis et s'il est conforme aux restrictions impos�es par l'utilisateur
	*
	* @var array $array		L'array dans lequel chercher et v�rifier la validit� des valeurs (le plus souvent $_POST ou $_GET)
	* @return bool
	*/
	public function isValid($array)
	{
		if(empty($this->elements) OR !$this->isSubmitted()) return FALSE;
		
		foreach($this->elements as $obj)
		{
			if(!$obj->isValid($array))
			{
				return FALSE;
			}
		}
		
		return TRUE;
	}
	
	/**
	* Modifie l'attribut indiquant si la protection contre le HTML est activ�e ou non
	*
	* @var bool $protect
	* @return bool
	*/
	public function protectHtml($protect)
	{
		return ($this->protectHtml = (bool) $protect);
	}
	
	/**
	* 
	* Appelle la m�thode getError de chaque objet TFormElement stock� dans le tableau element
	*
	* @return string		Les erreurs rencontr�es � la soumission du formulaire
	*/
	public function errors()
	{
		$out = '';
		foreach($this->elements as $obj)
		{
			$out .= $obj->getError();
		}
		return $out;
	}
	
	/**
	*
	* Pr�-rempli les champs du formulaires avec les donn�es de l'array
	* @var array array		L'array d'o� les valeurs sortent (le plus souvent $_POST ou $_GET)
	*/
	public function pre(Array $array)
	{
		foreach($this->elements as $obj)
		{
			$var = @$array[$obj->getName()];
			if(isset($var))
			{
				$obj->value($var);
				$obj->defaultChoice($var);
			}
		}
	}
		
	/**
	*
	* Pr�-rempli les champs du formulaires avec les donn�es d'un object
	* @var object
	*/
	public function preObject($modelObject){
		foreach($this->elements as $obj)
		{
			$prop = $obj->getName();
			
			$var = $modelObject->$prop;
			if(isset($var))
			{
				$obj->value($var);
				$obj->defaultChoice($var);
			}
		}
	}
	
	
	/** 
	*
	* Appelle la m�thode getError de chaque objet TFormElement stock� dans le tableau element
	*
	* @multiple var string/array 		Les noms des champs dont la valeur est � r�cup�rer
	* @return array/string					Array des valeurs concern�es, string si une seule valeur demand�e
	*/
	public function get_cleaned_data($param)
	{
		$cleaned_data = array();
		$out = '';
		
		if($this->protectHtml)
		{
			$_POST = array_map('htmlspecialchars', $_POST);
			$_GET = array_map('htmlspecialchars', $_GET);
		}
		
		if(is_array($param) && func_num_args()==1)
		{
			foreach($param as $name)
			{
				$cleaned_data[] = (isset($_POST[$name])) ? $_POST[$name] : NULL;
			}
		}
		elseif(!is_array($param) && func_num_args()==1)
		{
			return $_POST[$param];
		}
		else
		{
			foreach(func_get_args() as $name)
			{
				$cleaned_data[] = (isset($_POST[$name])) ? $_POST[$name] : NULL;
			}
		}
		return $cleaned_data;
	}
	
	/** 
	*
	* Renvoie chaque element du formulaire
	*
	* @return array			Array des valeurs concern�es
	*/
	public function getElement(){
		return $this->elements;
	}
	
	
	/**
	 * Attribut une classe CSS
	 */
	public function setClassCSS($pClassCSS){
		$this->classCSS = $pClassCSS;
	}

	
	
}